/* From input file "JUGGLE.text" */

#include <stdio.h>
#include <string.h>

#include "juggle.h"
 
/****************************************************************************
 * Written by Allen Knutson [allenk@through.ugcs.caltech.edu]        6/88   *
 *  Things to be added:                                                     *
 *      Several patterns listed on screen, with switching between them.     *
 *      Integrate the pattern maker LISTALLP that's in /src/games.          *
 *                                                                          *
 * Notes:                                                                   *
 *         I'm assuming that hold time = time of a 1-throw; theta = 1/2.    *
 *         Thus n-throw + hold time =   n      * (1-throw + hold time),     *
 *                                             as n-throws are defined,     *
 *              n-throw             = (2n-1)   *  1-throw.                  *
 *         Thus n-throw height      = (2n-1)~2 *  1-throw height,           *
 *                                   inducing the calculation above.        *
 ****************************************************************************/
 
/****************************************************************************
 *                                                                          *
 * Various mods by J. Gaudreau:                                             *
 *  7-May-1991 : Reformatted everything to make it "consistent."            *
 * 30-Jun-1991 : Added eXtensions.                                          *
 *                                                                          *
 ****************************************************************************/
 
/**************************************************************************
                                                                         
 Viciously hacked by Alastair Reid <areid@uk.ac.glasgow.dcs>             
 16-Sep-1991 : Keyboard operation partially restored                     
               Default value of detail adjusted to suit speed of SPARC   
               Slight changes made to use "virtual window"               
 17-Sep-1991 : Variable manipulation extensively modified because I      
               couldn't understand it.                                   
               Maximum value of detail adjusted to suit SPARC            
 18-Sep-1991 : Adapted towards my porgramming style and comments added   
               so that I can understand it.                              
               Major stylistic changes:                                  
                 I like 0..i-1 instead of 1..i                           
                 I dislike ++, --, -=, etc.                              
                 I dislike WITH=&some_var ... WITH->x                    
 21-Sep-1991 : Replaced part of animation routine with new parabola
               routine that allows for different hand heights. Gravity
               introduced to replace origina; scaling method. (Gravity
               probably needs more work to avoid arithmetic overflow.)
 23-Sep-1991 : Introduced hands explicitly - allowing them to be at
               different heights.
 10-Oct-1991 : Fixed a bug I'd added: startup sequences of odd length
               resulted in balls instantaneously switching hands shortly 
               after the end of the startup.
                                                                         
***************************************************************************/

/***************************************************************************
Lovingly hacked by John Robinson <jr@ksr.com>

 spring 1991 : added support for running with Unix curses(3) for dumb
               terminals.
  7-Nov-1991 : merged IBMPC, X and curses versions into a common source.
               formatted everything; diff'ing is hopeless now.
  8-Nov-1991 : add pause command: p or P.
 19-Nov-1991 : added private sleep routine jsleep() for Unix variants
               fixed initial display (showed balls before they were placed)
  3-Dec-1991 : added variable number of hands (1-10), set by -h switch
               made scoop per-hand; reduce initial value for large patterns
               added interactive step-to-next command: n/N
               for nhands != 2, fiddle last stepped-to hand only
               added -n switch for curses version to juggle "numbers";
                  default is letter O
               tweaked parabola() to avoid overflow for large numbers of balls
	          and/or large detail; thanks to jvert@microsoft.com
               added checking of patterns, plus automatic generation of
	          startup patterns if needed and none is provided
               made manifest constants for some of the hard-coded values
***************************************************************************/

long current; /* current option number */
long num_options; /* number of declared options */
long ballsize_option; /* to allow changing ballsize while juggling */

/* size of "virtual window" */
static int m_down, m_across;

hand hands[MAX_NUM_HANDS];
static worldvar fiddle[MAX_NUM_OPTIONS];
static ball world[MAX_PATTERN_LENGTH]; /* ball[0..] = state of balls */
static long pat[TOTAL_PATTERN_LENGTH];
static long m, c, i, j, detail;
static long nhands;
static boolean done, pausing, next_pause;
boolean do_numbers;
long gravity;
static long ticks, throwsnow;
static boolean emptyhand;
static long sum;
static boolean diagnos;
static int ch;
static long center, width, iscoop, highest;
static boolean onacid;
static long playing, now, prehighest, ballsize;

/*========================================================================*/
void redo_screen(void)
{
    int i;
 
    clrscr();
    m_colormode((long)M_XOR);
    for (i = 0; i < m; i++) {
	Draw_Ball(world[i].x, m_down - world[i].y, i, ballsize);
    }
    cleardevice();
}				/*redo_screen*/
 
 
/*long abs(long x){return( (x<0)?-x:x );}*/

long max(long x, long y){return( (x<y)?y:x );}
long min(long x, long y){return( (x<y)?x:y );}
long sqr(long x){return( x*x );}

/* sign(TRUE) == 1; sign(FALSE) == -1 */
long sign(boolean b){return( b?1:-1 );}

static void fiddle_around(short ch)
{
    gotoxy(1, 1);
    switch (ch) {
      case '!':
	system("");		/* ?? */
	redo_screen();
	break;
      case 'c':
      case 'C':
	pausing = 0;
	redo_screen();
	break;
      case 'a':
      case 'A':
	pausing = 0;
        onacid = 1;
	m_colormode((long)M_NORMAL);
        break;
      case 's':
      case 'S':
	redo_screen();
	pausing = 0;
        onacid = 0;
	m_colormode((long)M_XOR);
        redo_screen();
        break;
      case 't': 
	current = (current + 1) % num_options;
	printf("%s%5ld\n",fiddle[current].title,*fiddle[current].vptr);
	break;
      case 'T': 
	current = (current + num_options - 1) % num_options;
	printf("%s%5ld\n",fiddle[current].title,*fiddle[current].vptr);
	break;
      case 'f':
      case 'F':
	fiddle[current].new_ = fiddle[current].default_;
	if (!fiddle[current].slow)
	    *fiddle[current].vptr = fiddle[current].default_;
	printf("%s%5ld\n",fiddle[current].title,fiddle[current].new_);
	break;

      case '+':
      case '=':			/* synonym for + */
      case '-':
      case '_':			/* synonym for - */
	{long new_value, change;

	 if (ch == '+' || ch =='=')
	     change = fiddle[current].step;
	 else
	     change = - fiddle[current].step;

	 new_value = fiddle[current].new_ + change;

	 if (fiddle[current].bounded)
	     new_value = min(fiddle[current].upper,
			     max(fiddle[current].lower,new_value));

	 fiddle[current].new_ = new_value;

	 if (!fiddle[current].slow)
	     *fiddle[current].vptr = new_value;

	 printf("%s%5ld\n",fiddle[current].title,new_value);
     }
	break;

      case 'n':
      case 'N':
	pausing = 0;
	next_pause =  1;
	break;
      case 'p':
      case 'P':
	pausing = 1 - pausing;
	break;
      case 'q':
      case 'Q':
	pausing = 0;
        done = 1;
        break;
      default:
        printf("Unknown command %c\n",ch);
	break;
 
    }
 
}

static void setupgraphics(int argc, char *argv[])
{
    int g_driver = 0, g_mode = 0, g_error;

    detectgraph(&g_driver,&g_mode);
    if (g_driver < 0) {
	fprintf(stderr, "no graphics!\n");
	exit(-1);
    }
    if (g_driver || g_mode)
	fprintf(stderr, "found graphics driver #%d, mode #%d\n",
		g_driver,g_mode);
  
#ifdef XWINDOWS
    JballBegin(argc, argv);	/* parse X arguments if any; open display */
#else	/* XWINDOWS */
    initgraph(&g_driver,&g_mode,NULL);
#endif	/* XWINDOWS */
    g_error = graphresult();
    if (g_error < 0) {
	closegraph();
	fprintf(stderr, "Initgraph error: %s\n",grapherrormsg(g_error));
	exit(-1);
    }
    setbkcolor((long)BLACK);
}

void init_var(char title[256],
	      long *vptr,
	      boolean bounded,
	      long lower,
	      long upper,
	      boolean scaled,
	      long step,
	      boolean slow)
{
    strcpy(fiddle[num_options].title, title);
    fiddle[num_options].vptr = vptr;
    fiddle[num_options].new_ = *vptr;
    fiddle[num_options].bounded = bounded;
    fiddle[num_options].lower = lower;
    fiddle[num_options].upper = upper;
    fiddle[num_options].scaledep = scaled;
    fiddle[num_options].step = step;
    fiddle[num_options].slow = slow;
    fiddle[num_options].default_ = *vptr;
    gotoxy(1, num_options + 3);
    printf("%s [%5ld .. %5ld] %5ld\n", fiddle[num_options].title,
           fiddle[num_options].lower, fiddle[num_options].upper,
           *fiddle[num_options].vptr);
    num_options = num_options + 1;
}

static void select_fiddle_hand(int hand)
{
    static int first_option = -1;
    static char xbuff[11], ybuff[11], sbuff[11];
    long save_options;

    sprintf(xbuff, "Hand  %2d X", hand + 1);
    sprintf(ybuff, "Hand  %2d Y", hand + 1);
    sprintf(sbuff, "Scoop %2d  ", hand + 1);
    if (first_option < 0)	/* KLUDGE */
	first_option = num_options;
    save_options = num_options;
    num_options = first_option;
    init_var(xbuff,&hands[hand].x,    0,0,0,     0,10,0);
    init_var(ybuff,&hands[hand].y,    0,0,0,     0,10,0);
    init_var(sbuff,&hands[hand].scoop,0,0,0,     1, 5,0);
    num_options = save_options;
}

static void setup(void)
{
    long i;

    printf("%ld balls in the", m);
    for (i = 0; i < c; i++)
        printf(" %ld", pat[i + MAX_PATTERN_LENGTH]);
    printf(" pattern.\n");

    num_options = 0;
    init_var("Detail    ",&detail,        1,1,1000/m,0, 2,1);
    if (nhands == 2) {
	init_var("Left X    ",&hands[0].x,    0,0,0,     0,10,0);
	init_var("Left Y    ",&hands[0].y,    0,0,0,     0,10,0);
	init_var("Right X   ",&hands[1].x,    0,0,0,     0,10,0);
	init_var("Right Y   ",&hands[1].y,    0,0,0,     0,10,0);
	init_var("L Scoop   ",&hands[0].scoop,0,0,0,     1, 5,0);
	init_var("R Scoop   ",&hands[1].scoop,0,0,0,     1, 5,0);
    } else {
	select_fiddle_hand(0);
	num_options = num_options + 3; /* KLUDGE */
    }
    ballsize_option = num_options;
    init_var("Ballsize  ",&ballsize,      1,1,100,   1, 1,1);
    init_var("Gravity   ",&gravity,       1,-2000,-1,0, gravity/30-1,0);

    cleardevice();
#ifdef BITBLT
    /* setup bitblt images */
    for (i = 1; i <= m; i++) {
	setcolor(((i - 1) & 7) + 1); /* draw a ball */
	setfillstyle(SOLID_FILL,((i-1) & 7) + 1);
	pieslice(ballsize,ballsize, 0, 360, ballsize);
	world[i].bitmap= (void far *)malloc(imagesize(0,0,2*ballsize,2*ballsize));
	getimage(0,0,2*ballsize,2*ballsize,world[i].bitmap);
    }
#endif	/* BITBLT */

    current = 0;
}
 
long parabola(long a, long dy, long t, long T)
/* returns height of ball at time t/T thrown on path satisfying the
   following constraints.

   (0) y(t) = v * (t/T) + 0.5 * a * sqr(t/T)
       (ball thrown with initial velocity v in gravitational field a)
   (1) y(T) = dy
       (ball lands at height dy at time 1)

   Solving for v, we get

     y(T) = dy
   = {constraint 0}
     v * (T/T) + 0.5 * a * sqr(T/T) = dy
   = {arithmetic}
     v + 0.5 * a = dy
   = {algebra}
     v = dy - 0.5 * a

   This solution can be used to produce a calculation that strikes a
   balance between roundoff errors and overflows.

     y(t)
   = {constraint 0}
     v * (t/T) + 0.5 * a * sqr(t/T)
   = {v = dy - 0.5 * a}
     (dy - 0.5 * a) * (t/T) + 0.5 * a * sqr(t/T)
   = {arithmetic}
     ((2 * dy - a) * t * T + (a * sqr(t))) / (2 * sqr(T))

*/
/* Note that for dy = 0, y(t) is a*(t-T)*t / (2 * sqr(T)).
   The original code (which ignored dy) uses the following calculation

     (T-t) * t * 4 / sqr(detail) * scale/KLUDGE		[KLUDGE == 100]

   This suggests that an accelaration proportional to -sqr(throw) is
   appropriate.

from jvert@microsoft.com:
    return  t * ((a - 2 * dy) - (a * t) / T) / (2 * T);

that is close, but above rewrites as [opposite sign from jvert]:
   = {arithmetic}
     ((t * ((2 * dy - a)  + (a * t) / T)) / 2) / T

*/
{ return( ((t * ((2 * dy - a)  + (a * t) / T)) / 2) / T ); }

/* this calculation replaced by use of parabola */
/* long y_flight(long throw, long lat, long j)
/* /* j / detail == how far between sites we are */
/* /* throw = flight time 
/*    lat = amount of flight done */
/* {
/*   long howfar, throwfactor;
/* 
/*   howfar = (lat - 1) * detail + j;
/*   throwfactor = throw - 1;
/*   return( (scale*throwfactor - scale*howfar/detail) * howfar*4/detail/KLUDGE );
/* }
*/

void calculate_new_positions(ball b, long j, long *x, long *y)
/* b = ball
   j / detail == how far between sites we are
   2 * b.throw == flight time  (ie b.throw == height of throw)
   b.lateral == amount of flight done
   return new positions in x and y
*/
{
    long t,T;
    long dx,dy;
    long xdisp, ydisp, scoop;

    if (b.throw <=0){		/* ball is in hand */

	dx = - b.left_right	/* negate because b.lateral in <=0 */
	    * (3 * ballsize * ((b.lateral - 1) * detail + j))/detail;
	dy = 0;

    } else if (b.throw == nhands && nhands <= 2) {
	/* special case: hold in hand */

	dx = 0;
	dy = 0;

    } else if (b.lateral == b.throw * 2) { /* scooping */

	scoop = hands[b.handto].scoop;
	dx = hands[b.handto].x - hands[b.handfrom].x
	    + hands[b.handto].left_right * (scoop - (scoop * j)/detail);
	dy = hands[b.handto].y - hands[b.handfrom].y
	    + parabola( 5 * abs(scoop), 0, j, detail);

    } else {			/* in flight */

	xdisp = hands[b.handto].x - hands[b.handfrom].x
	    + hands[b.handto].left_right * hands[b.handto].scoop;
	ydisp = hands[b.handto].y - hands[b.handfrom].y;

	t = (b.lateral - 1)*detail + j; /* amount of flight so far */
	T = (2*b.throw-1)*detail; /* total flight time */
      
	dx = (xdisp * t)/T;
	dy = parabola( gravity*sqr(b.throw), ydisp, t, T );
    }

    *x = hands[b.handfrom].x + dx;
    *y = hands[b.handfrom].y + dy;

    return;
}

void calc_next_state(ball *b, long thrownow)
/* updates count "throwsnow" of total number of throws so far
   updates flag "emptyhand" to watch for hands being overfilled
   It might be useful to extend meaning of "thrownow" in something
   like the following way:
   -1 : pickup ball
   -2 : drop ball
*/
{
    switch (b->throw) {		/* never used by normal patterns? */
      case -2:
	printf("Can't drop them yet\n");
	break;
      case -1:			/* ball in hand - not yet thrown */
	if (b->lateral < 0) {	/* not ready to throw yet */
	    b->lateral += 1;	/* nudge it on */
	} else {		/* ball ready to be thrown from ball->handfrom */
	    b->lateral = 1;
	    b->throw = thrownow;
	    b->handto = (b->handfrom + b->throw) % nhands;
	    throwsnow = throwsnow+1;
	}
	break;
      case 0:			/* no throws left - catch ball */
	emptyhand = 1;
	break;
      default:			/* ball is in the pattern */
	if (b->lateral < b->throw*2) { /* in air */
	    b->lateral += 1;	/* nudge it on */
	    if (b->lateral == b->throw*2) {
		if (next_pause) { /* this is an interesting place to pause */
		    next_pause = 0;
		    pausing = 1;
		    if (nhands > 2)
			select_fiddle_hand(b->handto);
		}
	    }
	} else {		/* ball ready to be caught */
	    if (next_pause) {	/* this is an interesting place to pause */
		next_pause = 0;
		pausing = 1;
		if (nhands > 2)
		    select_fiddle_hand(b->handto);
	    }
	    b->lateral = 1;
	    b->throw = thrownow;
	    b->handfrom = b->handto; /* ball is where it last landed */
	    b->handto = (b->handfrom + b->throw) % nhands;
	    throwsnow = throwsnow+1;
	}
	break;
    }
    return;
}

void do_diagnostics()
{
    if (!diagnos)
	gotoxy(1, 6);
    if (diagnos) {
	for (i = m-1; i >= 0; i--) {
	    printf("%2d:%2d   ", world[i].throw, world[i].lateral);
	}
    }
    if (throwsnow != (ticks + 1 & 1) && !(throwsnow == 0 && pat[now+MAX_PATTERN_LENGTH] == 0)) {
	printf("\n     Bad pattern: caught %ld balls.", throwsnow); 
	diagnos = 1;
    }
    if (emptyhand) {
	printf("\n     Bad pattern: was holding a ball during a 0.");
	diagnos = 1;
    }
    if (diagnos) putchar('\n');
}

boolean mark_catch(unsigned long array[], long position)
{
    unsigned long mask;
    long offset;

    mask = 0x1L << (position % BITS_PER_LONG);
    offset = position / BITS_PER_LONG;
    if ((array[offset] & mask) != 0)
	return(0);		/* collision */
    array[offset] |= mask;
    return(1);
}

void check_pattern(long s)
{
    long pos, seq, held, limit, throw;
    unsigned long used[TOTAL_PATTERN_LENGTH/BITS_PER_LONG + 1];
    char *where;
    int i;

    for (i = 0; i < TOTAL_PATTERN_LENGTH/BITS_PER_LONG + 1; i++)
	used[i] = 0;
    held = m;
    limit = s + max(c, highest);
    for (pos = 0; pos < limit; pos++) {
	if (held > 0) {
	    if (!mark_catch(used, pos)) {
		closegraph();
		fprintf(stderr,
			"Oops, a ball lands while %d ball%s not yet airborne\n",
			held,
			held == 1 ? " is" : "s are");
		exit (-1);
	    }
	}
	if (pos < s)
	    throw = pat[MAX_PATTERN_LENGTH - s + pos];
	else
	    throw = pat[MAX_PATTERN_LENGTH + ((pos - s) % c)];
	if (throw != 0 || held <= 0) {	/* "empty" during launch means just hold */
	    held--;		/* a ball was really launched here */
	    if (!mark_catch(used, throw + pos)) {
		closegraph();
		if (pos < s) {
		    where = "startup";
		    seq = pos + 1;
		} else {
		    where = "pattern";
		    seq = pos + 1 - s;
		}
		if (throw == 0)
		    fprintf(stderr,
			    "Oops, an earlier throw hits an empty hand at %s throw %d\n",
			    where, seq);
		else
		    fprintf(stderr,
			    "Oops, %s throw %d (a %d) hits a hand holding an earlier throw\n",
			    where, seq, throw);
		exit (-1);
	    }
	}
    }

    for (pos = 0; pos < m; pos++) {
	if (mark_catch(used, pos + s)) {
	    closegraph();
	    fprintf(stderr,
		    "Oops, no throw lands in position %d after first cycle\n",
		    pos + 1);
	    exit (-1);
	}
    }
}

long find_init_pattern(void)
{
    long pos, seq, held, limit, throw;
    long s = 0;
    unsigned long used[TOTAL_PATTERN_LENGTH/BITS_PER_LONG + 1];
    char *where;
    int i;

    for (i = 0; i < TOTAL_PATTERN_LENGTH/BITS_PER_LONG + 1; i++)
	used[i] = 0;
    limit = c + highest;
    for (pos = 0; pos < limit; pos++) {
	throw = pat[MAX_PATTERN_LENGTH + (pos % c)];
	if (!mark_catch(used, throw + pos)) {
	    closegraph();
	    seq = pos + 1;
	    if (throw == 0)
		fprintf(stderr,
			"Oops, an earlier throw hits an empty hand at pattern throw %d\n",
			seq);
	    else
		fprintf(stderr,
			"Oops, pattern throw %d (a %d) hits a hand holding an earlier throw\n",
			seq, throw);
	    exit (-1);
	}
    }
    for (pos = limit - 1; pos >= 0 && pos >= m - s; pos--) {
	if (mark_catch(used, pos)) {
	    s++;
	    pat[MAX_PATTERN_LENGTH - s] = pos + s;
	}
    }

    if (s > 0) {
	printf("generated startup -s");
	for (pos = 0; pos < s; pos++) {
	    printf(" %d", pat[MAX_PATTERN_LENGTH + pos - s]);
	}
	printf("\n");
    }
    return(s);
}

#define strtol(a,b,c) atoi(a)

long read_pattern(int argc, char *argv[])
/* reads pattern and initialisation;
   returns the position of the first argument
     after the initialisation code. 
*/
{
    long patst,patfin, initst,ie, s, i;

    patst = 1;
    /* find first occurence of "-s" or "-x" or end of input */
    patfin = patst;
    while (patfin<argc && strcmp(argv[patfin], "-s")!=0 && strcmp(argv[patfin],"-x")!=0)
	patfin += 1;
    /* pattern is in argv[patst..patfin-1] */

    c = patfin - patst;		/* c = length of pattern */

    /* put pattern into pat[MAX_PATTERN_LENGTH..] and find largest & sum */
    highest = 0; sum = 0;
    for (i=0; i<c; i++) {
	pat[i+MAX_PATTERN_LENGTH] = strtol(argv[patst + i], NULL, 0);
	sum = sum + pat[i+MAX_PATTERN_LENGTH];
	highest = max(highest,pat[i+MAX_PATTERN_LENGTH]);
    }
    m = sum / c;
    /* pat[MAX_PATTERN_LENGTH..MAX_PATTERN_LENGTH+c-1] = pattern; m = number of balls */
    if (sum != m*c) {
	closegraph();
	fprintf(stderr,
		"Total of throws %d not divisible by number of throws %d\n",
		sum, c);
	exit (-1);
    }  
  
    /* skip -s if present */
    if ( (patfin<argc) && (strcmp(argv[patfin], "-s")==0) ) initst=patfin+1;
    else initst=patfin;
  
    /* find first occurence of "-x" or end of input */
    ie=initst; while ( (ie<argc) && strcmp(argv[ie],"-x")!=0 ) ie=ie+1;
    /* initialisation sequence is in argv[initst..ie-1] */
  
    s = ie - initst;
    prehighest=highest;
  
    for (i=0; i<s; i++) {
	pat[MAX_PATTERN_LENGTH-s + i] = strtol(argv[initst+i], NULL, 0);
	prehighest = max(prehighest,pat[MAX_PATTERN_LENGTH-s + i]);
    }

    if (s == 0)
	s = find_init_pattern();
    check_pattern(s);

    ticks=-2*s;

    /* pat[MAX_PATTERN_LENGTH-s..MAX_PATTERN_LENGTH-1] = initialisation sequence
       s = ie-initst (length of sequence)
       */
  
    /*
      all arguments from keyboard now read in 
      c = pattern length 
      m = number of balls
      pat[MAX_PATTERN_LENGTH..MAX_PATTERN_LENGTH+c-1] = pattern 
      pat[MAX_PATTERN_LENGTH-s..MAX_PATTERN_LENGTH-1] = initialisation sequence (of length s)
      ticks = - 2 * length of initialisation sequence
      */
  
    return(ie);
}

#ifdef IBMPC
/*
  m_ellipse(Tx, Ty, Tsize, Tsize, Tcolor);
*/

/*========================================================================*/
Draw_Ball(int Tx, int Ty, int Tcolor, int Tsize)
{
    setcolor((Tcolor - 1 & 7) + 1); /* draw new ball */
    circle(Tx, Ty, Tsize);
}

/*========================================================================*/
UnDraw_Ball(int Tx, int Ty, int Tcolor, int Tsize)
{
    setcolor(getbkcolor());	/* erase old ball */
    circle(Tx, Ty, Tsize);
}

/*========================================================================*/
AcidDraw_Ball(int Tx, int Ty, int Tcolor, int Tsize)
{
    /* identical to Draw_Ball() */
    setcolor((Tcolor - 1 & 7) + 1); /* draw new ball */
    circle(Tx, Ty, Tsize);
}
#endif	/* IBMPC */

main(int argc, char *argv[])
{
    int i, j;
    long tmp;
    long eop;			/* end of pattern and initialisation sequence */

    if (argc == 1) {
	fprintf(stderr,
		"Usage: juggle [-n] [-h <hands>] <pattern> [-s <pattern>]\n");
	exit(-1);
    }
    setupgraphics(argc, argv);
    m_across = getmaxx();
    m_down = getmaxy() - 10;	/* need some space for bitblt images */

    /* should check that argc>1 */
    /* diagnos = !strcmp(argv[1], "-d"); if (diagnos) {argc--;argv++;} */

    if (argc > 1 && !strcmp(argv[1], "-n")) {
	do_numbers = 1;		/* CURSES: juggle numbers, not O's */
	argc--; argv++;
    } else
	do_numbers = 0;

    nhands = 2;
    if (argc > 2 && !strcmp(argv[1], "-h")) {
	nhands = atoi(argv[2]);
	argc--; argc--; argv++; argv++;
	if (nhands <= 0 || nhands > MAX_NUM_HANDS) {
	    printf("Improper number of hands: %d\n", nhands);
	    exit(-1);
	}
    }

    eop = read_pattern(argc,argv);
    if ( (eop<argc) && (strcmp(argv[eop], "-x")==0) ) {
	argc = argc-eop;
	argv = argv+eop;
    }
  
    /*
      all arguments from keyboard now read in 
      c = pattern length 
      m = number of balls
      pat[MAX_PATTERN_LENGTH..MAX_PATTERN_LENGTH+c-1] = pattern 
      pat[MAX_PATTERN_LENGTH-s..MAX_PATTERN_LENGTH-1] = initialisation sequence (of length s=argc-patfin) 
      ticks = - 2 * length of initialisation sequence
      */
  
    iscoop = min(30 + (5 * m), 100);	/* wider initial scoop for more balls */
    width = (m_across - iscoop)/3;
    ballsize = 5;
    diagnos = 0;
    detail = 90/m + 2;
    /* if not using system clock to control speed, something inversely */
    /* proportional to m is a good choice. 400/m works well for SPARC */
    /* stations. */
  
    /* this calculation needs to be tweaked */
    /* an alternative is to build this scaling into xTens. That is, */
    /* xTens should allow control of magnification independently from */
    /* the scaling to cope with resizing windows. */
    gravity = (-6 * m_down /sqr(prehighest)) - 1;
  
    /* initial hand positions - the height needs be tweaked */
    for (i = 0; i < nhands-1; i += 2) {
	tmp = width - (width * i) / nhands;
	hands[i].x = m_across/2 - tmp;
	hands[i].y = 100;
	hands[i].scoop = iscoop;
	hands[i].left_right = -1; /* catch on outside, scoop in */
	/* ie catch on left of hand 0 and on right of hand 1 */

	hands[i + 1].x = m_across/2 + tmp;
	hands[i + 1].y = 100;
	hands[i + 1].scoop = iscoop;
	hands[i + 1].left_right = 1;
    }
    if (nhands % 2) {
	hands[nhands - 1].x = m_across/2;
	hands[nhands - 1].y = 100;
	hands[nhands - 1].scoop = iscoop;
	hands[nhands - 1].left_right = 1;
    }

    setup();			/* setup fiddlable variables */
    /* note: setup should only be called after all fiddlable variables
       have been initialised. */

    for (i = 0; i<m ; i=i+1 ) {
	world[i].handfrom = i%nhands; /* ball 0:left, 1:right, 2:left, etc. */
	world[i].left_right = sign(world[i].handfrom%2); /* queue on outside */
	world[i].lateral = -2*i; /* ball i will be thrown 2*i throws from now */
	world[i].throw = -1;	/* ball not yet thrown */
	world[i].ox = 0;
	world[i].oy = 0;
    }
  
    clrscr();
    cleardevice();
    /* m_choosecolors(1L); */
    onacid = 0;
    done = 0;
    pausing = 0;
    next_pause = 0;

    while (!done) {
	now = ticks / 2;
	if (now >= c)
	    now = now - c;
    
	/* calculate position of each ball after current throw */
	throwsnow = 0; emptyhand = 0; /* used for diagnostics */
	for (i = 0; i < m; i++)
	    calc_next_state(&world[i], pat[now + MAX_PATTERN_LENGTH]);
    
	/* print intermediate positions to get smooth animation */
	for (j = 0; !done && j < detail; j++) {

	    /* check for input events; loop on a pause */

	    do {
		if (kbhit())
		    fiddle_around((short)getch());
		cleardevice();
		if (pausing) {
		    /* ifdef'd out on IBMPCs! */
		    jsleep(30000); /* 30 milliseconds */
		}
	    } while (pausing);

	    /* calculate intermediate position */
	    for (i = 0; i < m; i++) {
		world[i].ox = world[i].x;
		world[i].oy = world[i].y;
		calculate_new_positions(world[i],j,&world[i].x,&world[i].y);
	    }

	    /* display on screen */
	    for (i = 0; i < m; i++) {
		if (world[i].x != world[i].ox || world[i].y != world[i].oy) {
		    if (onacid) {
#ifdef BITBLT
			putimage(world[i].x, m_down - world[i].y, world[i].bitmap, COPY_PUT);
#else	/* BITBLT */
			AcidDraw_Ball(world[i].ox, m_down - world[i].oy, i, fiddle[ballsize_option].new_);
#endif	/* BITBLT */
		    } else {
#ifdef BITBLT
			putimage(world[i].ox, m_down - world[i].oy, world[i].bitmap, XOR_PUT);
			putimage(world[i].x,  m_down - world[i].y,  world[i].bitmap, COPY_PUT);
#else	/* BITBLT */
			UnDraw_Ball(world[i].ox, m_down - world[i].oy, i, ballsize);
			Draw_Ball  (world[i].x,  m_down - world[i].y,  i, fiddle[ballsize_option].new_);
#endif	/* BITBLT */
		    }
		    world[i].ox = world[i].x;
		    world[i].oy = world[i].y;
		}
	    }

	    cleardevice();

	    /* ifdef'd out on IBMPCs! */
	    jsleep(10000);	/* 10 milliseconds */
	}			/* end of animation loop */

	for (i = 0; i < num_options; i++) {
	    if (fiddle[i].slow)
		*fiddle[i].vptr = fiddle[i].new_;
	}
 
	if (!diagnos)
	    gotoxy(1, 6);
	if (diagnos) {
	    for (i = m - 1; i >= 0; i--) {
		printf("%2d:%2d   ", world[i].throw, world[i].lateral);
	    }
	}
	do_diagnostics();
	ticks = (ticks + 1);
	if (ticks > 0)
	    ticks = ticks % (c * 4);
    }				/* end of main loop */
 
    /*	m_graphics_off(); */
    closegraph();

    exit(0);
}
 
/* detail, center, wingspan, scoop, height, ballsize */
/* juggling pattern */
/* number of balls, pattern length */
/* Height of a 1-throw */
/* How slow it is */
/* If you want to stop */
/* 1 per throw */
/* Throw to be made this tick */
/* 1 per tick iff the pattern works */
/* No hand should throw a 0-ball */
/* to count balls in CLI patterns */
/* where this ball starts */
/* half of this ball's intended displacement */
/* offset from right side */
/* each way */
/* the left hand catches at   */
/* center - width - scoop, */
/* and throws at              */
/* initialization */
/* traveling-towards-hand code */
/* There should be exactly 1 */
/* No actual ball should get a 0 */
/* End. */
/*eof*/
